// Importing necessary Mantine components and hooks
import { Menu, PasswordInput, rem, useMantineTheme } from '@mantine/core';

// Importing useFormik for form handling
import { useFormik } from 'formik';

// Importing hooks and state management tools
import { useEffect, useState } from 'react';
import { useDispatch } from 'react-redux';

// Importing Yup for form validation
import * as Yup from 'yup';

// Importing custom API function
import { postData } from '../../api/api';

// Importing custom components and Redux actions
// import {  Text, TextInput } from '../..';
import { setUserInfo } from '../../redux/actions/user';

// Importing type definitions
import { IconEyeCheck, IconEyeOff } from '@tabler/icons-react';
import axios from 'axios';
import { LuLockKeyhole, LuUserRound } from 'react-icons/lu';
import { useNavigate } from 'react-router-dom';
import { default as leftBackgroundImageSrc } from '../../assets/images/leftImage.png';
import backgroundImageSrc from '../../assets/images/login bg img.jpg';
import logo from '../../assets/images/muv-logo.svg';

import { Notification, Text, TextInput } from '../../components';
import { requestForToken } from '../../firebase/firebase.config';
import useScreenSize from '../../hooks/useScreenSize';
import { fetchAllData } from '../../redux/actions/reference';
import { inputStyle, inputStyle1 } from '../../theme/common.style';
import { LoginPageProps } from '../../types/login.type';
import { handleError } from '../../utils/common/function';

// Initial notification state
const initialNotification = {
  isSuccess: true, // Indicates whether the notification is successful
  isEnable: false, // Indicates whether the notification is enabled
  title: '', // Title of the notification
  message: '', // Message of the notification
};

// Preload the background image
const backgroundImage = new Image();
// eslint-disable-next-line @typescript-eslint/no-require-imports
backgroundImage.src = backgroundImageSrc;

const leftBackgroundImage = new Image();

// eslint-disable-next-line @typescript-eslint/no-require-imports
leftBackgroundImage.src = leftBackgroundImageSrc;

// LoginPage function component
export function LoginPage() {
  // Initialize theme hook
  const theme = useMantineTheme();

  const navigate = useNavigate();

  // Initialize dispatch for Redux actions
  const dispatch = useDispatch();

  const { isLargeScreen } = useScreenSize();

  // Initialize notification state
  const [notification, setNotification] = useState(initialNotification);
  const [isLoading, setIsLoading] = useState(false);

  const [userDetails] = useState();
  const [imagesLoaded, setImagesLoaded] = useState({
    background: false,
    leftBackground: false,
  });

  useEffect(() => {
    // Preload the background image
    const backgroundImage = new Image();
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    backgroundImage.src = backgroundImageSrc;

    const leftBackgroundImage = new Image();
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    leftBackgroundImage.src = leftBackgroundImageSrc;
    backgroundImage.onload = () => {
      setImagesLoaded(prev => ({ ...prev, background: true }));
    };
    leftBackgroundImage.onload = () => {
      setImagesLoaded(prev => ({ ...prev, leftBackground: true }));
    };
  }, []);

  // Initialize formik for form handling
  const formik = useFormik({
    initialValues: {
      email: '', // Initial email value
      password: '', // Initial password value
    },
    // Define form validation schema
    validationSchema: Yup.object({
      email: Yup.string().email('Invalid email').required('Required'),
      password: Yup.string().required('Required'),
    }),
    // Define form submission handler
    onSubmit: async (values: LoginPageProps) => {
      setIsLoading(true);
      // Post login data to the server
      const webFCMToken = await requestForToken();
      postData('/auth/login', {
        ...values,
        fcmTokenWeb: webFCMToken,
      })
        .then((res: any) => {
          if (res?.user?.roleId === '0') {
            handleError('Guest is not allowed to login.');
          } else {
            dispatch(setUserInfo(res));
            fetchAllData(dispatch);
          }
        })
        .catch(error => {
          handleError(error);
        })
        .finally(() => setIsLoading(false));
    },
  });

  // Handle notification close
  const handleNotificationClose = () => {
    setNotification(initialNotification);
  };

  const VisibilityToggleIcon = ({ reveal }: { reveal: boolean }) =>
    reveal ? (
      <IconEyeOff
        style={{
          width: '32px',
          height: '32px',
          color: '#ffffff',
        }}
      />
    ) : (
      <IconEyeCheck
        style={{
          width: '26px',
          height: '26px',
          color: '#ffffff',
        }}
      />
    );

  return imagesLoaded?.background && imagesLoaded.leftBackground ? (
    isLargeScreen ? (
      <>
        {notification.isEnable && (
          <Notification
            title={notification.title}
            withClose
            color={
              notification.isSuccess ? theme.colors.primaryColors[0] : 'red'
            }
            handleCloseNotification={handleNotificationClose}
          >
            <Text fw={500}>{notification.message}</Text>
          </Notification>
        )}

        <div className="flex justify-between items-stretch h-[100vh]">
          <div className="flex-1 flex justify-center items-center h-full">
            <video
              src={'/loginpage.mp4'}
              className="w-full h-full object-cover"
              autoPlay
              loop
              muted
              playsInline
            ></video>
          </div>
          <div
            className="relative flex-1 bg-cover bg-no-repeat bg-center bg-[#3B1E6D] h-screen items-center justify-center flex bg-gradient-to-br
from-[#563097]
via-[#3B1E6D]
to-[#563097]"
          >
            <div className="flex flex-col items-center w-full max-w-[500px] px-5">
              <img
                src={logo}
                alt=""
                className="max-h-full h-[92px] w-[226px] mb-[39px] mt-[60px]"
              />
              <div className="text-[#ffffff] text-[36px] font-bold font-productsans leading-[49.7px] mt-1">
                LOGIN
              </div>
              <div className="text-[#ffffff] text-[17px] font-normal font-productsans leading-[49.7px] -mt-5">
                Enter your email address to Login
              </div>
              <form
                className="w-full flex flex-col gap-4 mt-[42px]"
                onSubmit={formik.handleSubmit}
              >
                <TextInput
                  className="custom-input"
                  label="Email Address"
                  placeholder="Enter your email"
                  value={formik.values?.email}
                  leftSection={
                    <LuUserRound
                      style={{
                        width: '25px',
                        height: '25px',
                        color: '#FFFFFF',
                        position: 'absolute',
                        left: '25px',
                        top: '50%',
                        transform: 'translateY(-50%)',
                      }}
                    />
                  }
                  onChange={(value: any) =>
                    formik.setFieldValue('email', value)
                  }
                  radius="6px"
                  styles={inputStyle}
                  error={formik.touched.email && formik.errors.email}
                />

                <PasswordInput
                  label="Password"
                  placeholder="Enter your password"
                  value={formik.values?.password}
                  visibilityToggleIcon={VisibilityToggleIcon}
                  className="custom-password-input custom-input"
                  radius="6px"
                  styles={inputStyle1}
                  leftSection={
                    <LuLockKeyhole
                      style={{
                        width: '25px',
                        height: '25px',
                        color: '#FFFFFF',
                        position: 'absolute',
                        left: '25px',
                        top: '50%',
                        transform: 'translateY(-50%)',
                      }}
                    />
                  }
                  onChange={event =>
                    formik.setFieldValue('password', event.target.value)
                  }
                  error={formik.touched.password && formik.errors.password}
                  type="password" // Ensure the input type is password to show asterisks
                />

                <div className="flex items-end justify-end w-full -mt-2">
                  <div
                    onClick={() => navigate('/forgot-password')}
                    className="cursor-pointer underline font-productsans text-[#ffffff] text-[16px] font-normal"
                  >
                    Forgot Password?
                  </div>
                </div>
                <div className="mt-[2px] flex flex-col items-center">
                  <button
                    className="w-[200px] h-[50px] cursor-pointer select-none bg-[#D83F87] py-2 px-5 text-center align-middle text-sm text-color-white shadow font-productsans rounded-md text-[18px] font-bold"
                    type="submit"
                    disabled={isLoading}
                  >
                    {isLoading ? 'Loading' : 'Sign In'}
                  </button>
                </div>
              </form>{' '}
              <Menu
                width={200}
                position="right"
                transitionProps={{ transition: 'pop-top-right' }}
              >
                <Menu.Target>
                  <button className="mt-2 text-[#fff] text-sm font-normal hover:underline cursor-pointer">
                    Register as
                  </button>
                </Menu.Target>

                <Menu.Dropdown>
                  <Menu.Item
                    onClick={() => navigate('/business-register')}
                    styles={{
                      item: {
                        fontSize: rem(14),
                        '&:hover': {
                          backgroundColor: 'var(--mantine-color-gray-1)',
                        },
                      },
                    }}
                  >
                    Business
                  </Menu.Item>
                  <Menu.Item
                    onClick={() => navigate('/entertainer-register-page')}
                    styles={{
                      item: {
                        fontSize: rem(14),
                        '&:hover': {
                          backgroundColor: 'var(--mantine-color-gray-1)',
                        },
                      },
                    }}
                  >
                    Entertainer
                  </Menu.Item>
                </Menu.Dropdown>
              </Menu>
            </div>
          </div>
        </div>
      </>
    ) : (
      <>
        {notification.isEnable && (
          <Notification
            title={notification.title}
            withClose
            color={
              notification.isSuccess ? theme.colors.primaryColors[0] : 'red'
            }
            handleCloseNotification={handleNotificationClose}
          >
            <Text fw={500}>{notification.message}</Text>
          </Notification>
        )}

        <div className="flex flex-col lg:flex-row justify-between items-stretch h-auto">
          {/* Right Form Section */}
          <div
            className="flex-1 bg-cover bg-no-repeat bg-center h-auto lg:h-screen min-h-screen flex items-center justify-center bg-[#3B1E6D] bg-gradient-to-br
from-[#563097]
via-[#3B1E6D]
to-[#563097] "
          >
            <div className="flex flex-col items-center w-full max-w-[500px] px-5">
              {/* Logo */}
              <img
                src={logo}
                alt="Logo"
                className="max-h-[80px] w-auto mb-5 mt-[20px] "
              />
              {/* Title */}
              <h1 className="text-[#ffffff] text-[28px] sm:text-[36px] font-bold font-productsans text-center">
                LOGIN
              </h1>
              {/* Description */}
              <p className="text-[#ffffff] text-[14px] sm:text-[17px] font-normal font-productsans mb-8 text-center">
                Enter your email address to Login
              </p>
              {/* Form */}
              <form
                className="w-full flex flex-col gap-4"
                onSubmit={formik.handleSubmit}
              >
                {/* Email Input */}
                <TextInput
                  className="custom-input"
                  label="Email Address"
                  placeholder="Enter your email"
                  value={formik.values?.email}
                  leftSection={
                    <LuUserRound
                      style={{
                        width: '29.26px',
                        height: '32.51px',
                        color: '#FFFFFF',
                      }}
                    />
                  }
                  onChange={(value: any) =>
                    formik.setFieldValue('email', value)
                  }
                  radius="8px"
                  styles={inputStyle}
                  error={formik.touched.email && formik.errors.email}
                />

                {/* Password Input */}
                <PasswordInput
                  className="custom-input"
                  label="Password"
                  placeholder="Enter your password"
                  value={formik.values?.password}
                  visibilityToggleIcon={VisibilityToggleIcon}
                  radius="8px"
                  styles={inputStyle1}
                  leftSection={
                    <LuLockKeyhole
                      style={{
                        width: '24px',
                        height: '24px',
                        color: '#FFFFFF',
                      }}
                    />
                  }
                  onChange={event =>
                    formik.setFieldValue('password', event.target.value)
                  }
                  error={formik.touched.password && formik.errors.password}
                />

                {/* Forgot Password */}
                <div className="flex items-end justify-end w-full">
                  <div
                    onClick={() => navigate('/forgot-password')}
                    className="cursor-pointer font-productsans text-[#ffffff] mt-[6px] text-[14px] sm:text-[17px] font-normal"
                  >
                    Forgot Password?
                  </div>
                </div>

                {/* Submit Button */}
                <div className="mt-[32px] flex flex-col items-center">
                  <button
                    className="w-[200px] h-[50px] cursor-pointer select-none bg-[#D83F87] py-2 px-5 text-center align-middle text-sm text-color-white shadow font-productsans rounded-md text-[16px] sm:text-[18px] font-bold"
                    type="submit"
                    disabled={isLoading}
                  >
                    {isLoading ? 'Loading' : 'Sign In'}
                  </button>

                  {/* Register Link */}
                  <div className="mt-[12px] text-center"></div>
                </div>
              </form>{' '}
              <Menu
                width={200}
                position="right"
                transitionProps={{ transition: 'pop-top-right' }}
              >
                <Menu.Target>
                  <button className="mt-2 text-[#fff] text-sm font-normal hover:underline cursor-pointer">
                    Register as
                  </button>
                </Menu.Target>

                <Menu.Dropdown>
                  <Menu.Item
                    onClick={() => navigate('/business-register')}
                    styles={{
                      item: {
                        fontSize: rem(14),
                        '&:hover': {
                          backgroundColor: 'var(--mantine-color-gray-1)',
                        },
                      },
                    }}
                  >
                    Business
                  </Menu.Item>
                  <Menu.Item
                    onClick={() => navigate('/entertainer-register-page')}
                    styles={{
                      item: {
                        fontSize: rem(14),
                        '&:hover': {
                          backgroundColor: 'var(--mantine-color-gray-1)',
                        },
                      },
                    }}
                  >
                    Entertainer
                  </Menu.Item>
                </Menu.Dropdown>
              </Menu>
            </div>
          </div>
        </div>
      </>
    )
  ) : null;
}
