import {
  Avatar,
  Burger,
  Grid,
  Indicator,
  Menu,
  rem,
  useMantineTheme,
} from '@mantine/core';
import {
  MenuItem,
  OutlinedInput,
  Select,
  SelectChangeEvent,
} from '@mui/material';
import { IconChevronDown } from '@tabler/icons-react';
import { useEffect, useState } from 'react';
import { FaBell, FaCaretDown } from 'react-icons/fa';
import { IoIosLogOut } from 'react-icons/io';
import { useDispatch, useSelector } from 'react-redux';
import { Outlet, useNavigate } from 'react-router-dom';
import { fetchData, putData } from '../../api/api';
import bh from '../../assets/images/bh.webp';
import gb from '../../assets/images/gb.png';
import useDisclosure from '../../hooks/useDisclosure';
import useScreenSize from '../../hooks/useScreenSize';
import { Navbar } from '../../layout';
import { clearUserInfo, setLanguage } from '../../redux/actions/user';
import { SET_CURRENCY } from '../../redux/constants/user';

type UserNotification = {
  isRead: boolean;
};

type Notification = {
  userNotifications?: UserNotification[];
};

const languageData = [
  {
    value: 'en',
    label: 'EN',
    flag: (
      <div>
        <img src={gb} alt="us flag" className="w-6 h-4" />
      </div>
    ),
  },
  {
    value: 'ar',
    label: 'AR',
    flag: (
      <div>
        <img src={bh} alt="Arabic" className="w-6 h-4" />
      </div>
    ),
  },
];

function AppShell() {
  const [opened, { toggle }] = useDisclosure();
  const theme = useMantineTheme();
  const { isSmallScreen, isMediumScreen } = useScreenSize();
  const { referenceData } = useSelector((state: any) => state?.referenceData);
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const navigate = useNavigate();
  const dispatch = useDispatch();
  const { userInfo, preferences } = useSelector(
    (state: any) => state?.userInfo
  );
  const [currencyId, setCurrencyId] = useState('1');
  const [loading, setLoading] = useState(false);
  const [unreadCount, setUnreadCount] = useState(0);

  const getDirection = (language: string): 'rtl' | 'ltr' => {
    const rtlLanguages = ['ur', 'ar', 'he', 'fa'];
    return rtlLanguages.includes(language) ? 'rtl' : 'ltr';
  };

  const handleChange = (event: SelectChangeEvent<string>) => {
    const newLanguage = event.target.value;
    dispatch(setLanguage(newLanguage));
    const direction = getDirection(newLanguage);
    document.documentElement.setAttribute('dir', direction);
  };

  const handleCurrencyChange = async (e: any) => {
    const selectedId = e.target.value;
    setCurrencyId(selectedId);
    const selectedCurrency = referenceData?.currencies.find(
      (c: any) => c.id === selectedId
    );

    if (!selectedCurrency || !userInfo?.id) return;

    setLoading(true);

    try {
      await Promise.all([
        putData(`/user-preferences/${userInfo.id}`, {
          preferredCurrencyId: selectedId,
        }),
        putData(`/users/${userInfo.id}`, {
          currencyId: selectedId,
        }),
      ]);

      dispatch({
        type: SET_CURRENCY,
        payload: {
          preferredCurrencyId: selectedCurrency.id,
          details: selectedCurrency,
        },
      });
    } catch (error) {
      console.error('Currency update error:', error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (!userInfo?.id) return;

    const fetchUnreadCount = () => {
      fetchData(`notifications/unread-count/${userInfo.id}`)
        .then((res: any) => setUnreadCount(res.unreadCount))
        .catch(error => console.error('Error:', error));
    };

    fetchUnreadCount();

    const interval = setInterval(fetchUnreadCount, 10000);

    return () => clearInterval(interval);
  }, [userInfo?.id]);

  const renderMobileLayout = () => (
    <div className="bg-[#3B1E6D]">
      <Grid className="flex flex-row pt-4 justify-between bg-container rounded-b-2xl">
        <Grid.Col span={2}>
          <div className="mt-1.5 ml-2">
            <Burger
              opened={opened}
              onClick={toggle}
              aria-label="Toggle navigation"
              color="#ffffff"
            />
          </div>
        </Grid.Col>
        <Grid.Col span={10}>
          <div className="flex items-center justify-end gap-2 pr-2">
            {renderHeaderControls(true)}
          </div>
        </Grid.Col>
      </Grid>
      {opened ? (
        <div className="h-screen">
          <Navbar onClick={toggle} />
        </div>
      ) : (
        <Outlet />
      )}
    </div>
  );

  const renderDesktopLayout = () => (
    <div className="flex h-screen">
      {/* Sidebar */}
      <div className="w-56 sticky top-0 font-productsans bg-container h-full overflow-y-auto">
        <Navbar onClick={() => {}} />
      </div>

      {/* Main Content */}
      <div className="flex-1 flex flex-col">
        <div className="flex justify-between items-center px-8 py-4 bg-[#452975] h-20">
          <div className="flex-1" /> {/* Spacer */}
          <div className="flex items-center gap-4">
            {renderHeaderControls(false)}
          </div>
        </div>
        <div className="flex-1 overflow-y-auto">
          <Outlet />
        </div>
      </div>
    </div>
  );

  const renderHeaderControls = (isMobile: boolean) => (
    <>
      <div className={isMobile ? 'w-24' : 'w-28'}>
        <Select
          labelId="language-select-label"
          id="language-select"
          value={language}
          onChange={handleChange}
          displayEmpty
          size={isMobile ? 'small' : 'medium'}
          className="w-full"
          IconComponent={props => (
            <IconChevronDown
              {...props}
              style={{
                width: rem(16),
                height: rem(16),
                color: '#ffffff',
              }}
            />
          )}
          input={
            <OutlinedInput
              sx={{
                '& .MuiOutlinedInput-notchedOutline': {
                  borderColor: 'rgba(255, 255, 255, 0.08)',
                  borderRadius: '8px',
                },
                '&:hover .MuiOutlinedInput-notchedOutline': {
                  borderColor: 'rgba(255, 255, 255, 0.08)',
                },
                '&.Mui-focused .MuiOutlinedInput-notchedOutline': {
                  borderColor: 'rgba(255, 255, 255, 0.05)',
                },
                color: '#fff',
              }}
            />
          }
        >
          {languageData?.map((lang: any) => (
            <MenuItem key={lang.value} value={lang.value}>
              <div className="flex items-center gap-2 font-productsans">
                {lang.flag}
                <span>{lang.label}</span>
              </div>
            </MenuItem>
          ))}
        </Select>
      </div>

      <div className={isMobile ? 'w-24' : 'w-28'}>
        <Select
          labelId="currency-select-label"
          id="currency-select"
          value={preferences.currency?.preferredCurrencyId || '1'}
          onChange={async e => await handleCurrencyChange(e)}
          size={isMobile ? 'small' : 'medium'}
          className="w-full"
          IconComponent={props => (
            <FaCaretDown
              {...props}
              style={{
                width: rem(18),
                height: rem(18),
                color: '#D83F87',
              }}
            />
          )}
          input={
            <OutlinedInput
              sx={{
                '& .MuiOutlinedInput-notchedOutline': {
                  borderColor: 'rgba(255, 255, 255, 0.0)',
                  borderRadius: '8px',
                },
                '&:hover .MuiOutlinedInput-notchedOutline': {
                  borderColor: 'rgba(255, 255, 255, 0.0)',
                },
                '&.Mui-focused .MuiOutlinedInput-notchedOutline': {
                  borderColor: 'rgba(255, 255, 255, 0.0)',
                },
                color: '#fff',
              }}
            />
          }
        >
          {referenceData?.currencies.map((currency: any) => (
            <MenuItem key={currency.id} value={currency.id}>
              <div className="flex items-center gap-1 font-productsans">
                ({currency.symbol}) {currency.code}
              </div>
            </MenuItem>
          ))}
        </Select>
      </div>

      <div className="relative flex items-center mr-4">
        {unreadCount > 0 ? (
          <Indicator
            inline
            size={19}
            offset={4}
            color="#FF69B4"
            withBorder
            label={unreadCount}
            className="rounded-full custom-header"
          >
            <FaBell
              className="align-middle select-none text-center transition-all cursor-pointer"
              color="#ffffff"
              style={{ height: 23.09, width: 22 }}
              onClick={() => navigate('/notification')}
            />
          </Indicator>
        ) : (
          <FaBell
            className="align-middle select-none text-center transition-all cursor-pointer custom-header"
            color="#ffffff"
            style={{ height: 23.09, width: 22 }}
            onClick={() => navigate('/notification')}
          />
        )}
      </div>

      <Menu
        shadow="md"
        width={100}
        {...(!isMobile && {
          trigger: 'hover',
          openDelay: 50,
          closeDelay: 200,
        })}
      >
        <Menu.Target>
          <div className="flex items-center gap-2 cursor-pointer">
            <Avatar
              src={userInfo?.profilePicture || null}
              size={isMobile ? 36 : 42}
              radius="xl"
              color="gray"
              className="border-2 border-gray-400"
            >
              {userInfo?.fullNameOrBusinessName?.[0] || ''}
            </Avatar>

            {!isMobile && (
              <>
                <div className="text-[#FFF] text-sm whitespace-nowrap overflow-hidden text-ellipsis max-w-[120px]">
                  {userInfo?.fullNameOrBusinessName || 'N/A'}
                </div>
                <IconChevronDown
                  style={{
                    color: '#ffffff',
                    height: rem(15),
                    width: rem(15),
                  }}
                />
              </>
            )}
          </div>
        </Menu.Target>
        <Menu.Dropdown>
          <Menu.Item
            rightSection={<IoIosLogOut size={18} />}
            onClick={() => dispatch(clearUserInfo())}
          >
            <div className="font-normal text-sm text-black flex items-center justify-center cursor-pointer">
              Logout
            </div>
          </Menu.Item>
        </Menu.Dropdown>
      </Menu>
    </>
  );

  return (
    <div>
      {isSmallScreen || isMediumScreen
        ? renderMobileLayout()
        : renderDesktopLayout()}
    </div>
  );
}

export default AppShell;
