import {
  Avatar,
  Box,
  Button,
  Group,
  Loader,
  Paper,
  Text,
  Title,
} from '@mantine/core';
import { useEffect, useState } from 'react';
import { AiOutlineArrowLeft } from 'react-icons/ai';
import { useNavigate, useSearchParams } from 'react-router-dom';
import { fetchData, putData } from '../../api/api';
import OnlyPagination from '../../components/table/onlypagination';
import { paginationInfoValue } from '../../utils/common/constant.objects';
import {
  extractPageInfo,
  handleError,
  handleSuccess,
  removeEmptyValueFilters,
} from '../../utils/common/function';
import { handlePaginationValue } from '../../utils/common/pagination.Helper';
import { useSelector } from 'react-redux';
import { getTranslation } from '../../layout/languages';

const AngelWings = () => {
  const navigate = useNavigate();
  const [angelWingsData, setAngelWingsData] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [refreshTable, setRefreshTable] = useState(true);
  const [paginationInfo, setPaginationInfo] = useState(paginationInfoValue);
  const [searchParams, setSearchParams] = useSearchParams();
  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const fetchAngelWings = async () => {
    setLoading(true);
    try {
      const filters = removeEmptyValueFilters([
        {
          field: 'angel_wings.businessId',
          operator: 'eq',
          value: userInfo?.id,
        },
        {
          field: 'angel_wings.isResolved',
          operator: 'neq',
          value: 'true',
        },
      ]);
      const filterObject = JSON.stringify({ filter: filters });
      const queryParams = new URLSearchParams({
        rpp: paginationInfo.rowPerPage.toString(),
        page: (paginationInfo.currentPage === 0
          ? 1
          : paginationInfo.currentPage
        ).toString(),
        filter: filterObject,
      });

      const url = `/angel-wings?${queryParams.toString()}`;
      const res: any = await fetchData(url);

      setAngelWingsData(res.data);
      const getPages = extractPageInfo(res.pages);
      setPaginationInfo({
        ...paginationInfo,
        totalRecords: res.total,
        totalPages: getPages?.totalPages ?? 0,
      });
    } catch (error) {
      console.error('Error fetching Angel Wings data:', error);
      handleError(error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchAngelWings();
  }, [paginationInfo.currentPage, refreshTable]);

  const handleResolve = (id: string) => {
    setLoading(true);
    putData(`/angel-wings/${id}/resolve`, {})
      .then(() => {
        handleSuccess('Ticket resolved successfully');
        setRefreshTable(prev => !prev);
      })
      .catch((error: any) => {
        handleError(error);
      })
      .finally(() => {
        setLoading(false);
      });
  };

  const handleDelete = (id: string) => {
    setLoading(true);
    putData(`/angel-wings/${id}/delete`, {})
      .then(() => {
        handleSuccess('Ticket deleted successfully');
        setRefreshTable(prev => !prev);
      })
      .catch((error: any) => {
        handleError(error);
      })
      .finally(() => {
        setLoading(false);
      });
  };

  const handlePagination = (actionType: string, value?: any) => {
    handlePaginationValue(
      actionType,
      value,
      searchParams,
      paginationInfo,
      setPaginationInfo,
      setSearchParams
    );
  };

  return (
    <Paper
      radius={0}
      style={{
        background: 'radial-gradient(#6534BA 43%, #3B1E6D)',
        minHeight: '100vh',
        width: '100%',
        alignItems: 'center',
        justifyContent: 'center',
        flexDirection: 'column',
        padding: '40px 20px',
        paddingRight: '4%',
        paddingLeft: '4%',
      }}
    >
      <Group
        style={{
          justifyContent: 'space-between',
          alignItems: 'center',
          marginBottom: '30px',
        }}
      >
        <Group>
          <AiOutlineArrowLeft
            size={22}
            color="white"
            onClick={() => navigate(-1)}
          />
          <Title
            style={{
              color: 'white',
              fontSize: '28px',
              fontWeight: 'bold',
              fontFamily: 'Product Sans',
            }}
          >
            {getTranslation(language, 'navbar', 'Angel Wings')}
          </Title>
        </Group>
      </Group>

      <div
        className="rounded-2xl font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
        style={{ height: 'auto', width: '100%', minHeight: '450px' }}
      >
        {loading ? (
          <div className="flex items-center justify-center h-full mt-36">
            <Loader size="lg" color="#C28CFF" />
          </div>
        ) : angelWingsData.length === 0 ? (
          <Text
            style={{
              color: 'white',
              fontSize: '16px',
              fontFamily: 'Product Sans',
              textAlign: 'center',
              marginTop: '50px',
            }}
          >
            {getTranslation(
              language,
              'angelwing',
              'No angel wings tickets found.'
            )}
          </Text>
        ) : (
          angelWingsData.map((item, index) => (
            <Paper
              key={item.id}
              withBorder
              p="xl"
              mb="2.5%"
              mt="2%"
              style={{
                borderRadius: '6px',
                background: 'rgba(255, 255, 255, 0.1)',
                backdropFilter: 'blur(2px)',
                padding: '29px',
                border: '1px solid rgba(255, 255, 255, 0.3)',
                minHeight: '150px',
                width: '98%',
                margin: '0 auto',
                display: 'flex',
                flexDirection: 'column',
                justifyContent: 'space-between',
                fontFamily: 'Product Sans',
              }}
              className="px-4 py-4"
            >
              <div className="flex flex-row justify-between">
                <div className="flex flex-row gap-4">
                  <Avatar
                    src={item?.user?.profile?.profilePicture || null}
                    alt="User"
                    size={50}
                    radius="xl"
                    color="gray"
                  >
                    {item?.user?.profile?.fullNameOrBusinessName[0] || ''}
                  </Avatar>

                  <div className="flex flex-col">
                    <Group
                      gap="md"
                      align="center"
                      style={{ justifyContent: 'flex-start' }}
                    >
                      <Text
                        style={{
                          color: 'white',
                          fontSize: '18px',
                          flex: 1,
                          textAlign: 'left',
                        }}
                        className="font-productsans"
                      >
                        {item.title}
                      </Text>
                    </Group>

                    <Text
                      style={{
                        color: 'white',
                        fontSize: '14px',
                        textAlign: 'left',
                        marginTop: '8px',
                      }}
                      className="font-productsans"
                    >
                      {item.description
                        .split(/(https:\/\/[^\s]+)/g)
                        .map((part: any, index: any) =>
                          part.startsWith('https://') ? (
                            <a
                              key={index}
                              href={part}
                              target="_blank"
                              rel="noopener noreferrer"
                              className="underline"
                            >
                              {part}
                            </a>
                          ) : (
                            part
                          )
                        )}
                    </Text>

                    <Box mt="xs">
                      <Group
                        style={{
                          color: 'white',
                          fontSize: '14px',
                          fontWeight: 'bold',
                          fontFamily: 'Product Sans',
                          justifyContent: 'flex-start',
                        }}
                      >
                        {item.user?.phoneNumber && (
                          <Text>{item.user?.phoneNumber ?? ''}</Text>
                        )}
                        {item.user?.phoneNumber && <Text>|</Text>}

                        <Text>
                          {getTranslation(
                            language,
                            'angelwing',
                            'Ticket Number:'
                          )}
                          {item.ticketNumber ?? '04586'}
                        </Text>
                        <Text>|</Text>
                        <Text style={{ textDecoration: 'underline' }}>
                          {getTranslation(
                            language,
                            'angelwing',
                            'Emergency Contact:'
                          )}
                          {item.user?.emergencyContacts?.length > 0
                            ? item.user.emergencyContacts
                                .map(
                                  (contact: { phoneNumber: any }) =>
                                    contact.phoneNumber
                                )
                                .filter(Boolean)
                                .join(', ')
                            : 'N/A'}
                        </Text>
                      </Group>
                    </Box>
                  </div>
                </div>

                <div className="flex flex-row gap-2">
                  <Button
                    style={{
                      backgroundColor: '#418F90',
                      color: 'white',
                    }}
                    disabled={item.isResolved}
                    onClick={() => handleResolve(item.id)}
                  >
                    {getTranslation(
                      language,
                      'angelwing',
                      `${item.isResolved ? 'Resolved' : 'Resolve'}`
                    )}
                  </Button>
                  <Button color="red" onClick={() => handleDelete(item.id)}>
                    {getTranslation(language, 'angelwing', 'Delete')}
                  </Button>
                </div>
              </div>
            </Paper>
          ))
        )}
      </div>

      {!loading && angelWingsData.length > 0 && (
        <div className="ml-6">
          <OnlyPagination
            data={angelWingsData}
            columns={[]}
            isLoading={false}
            paginationInfo={paginationInfo}
            handlePagination={handlePagination}
          />
        </div>
      )}
    </Paper>
  );
};

export default AngelWings;
