import { initializeApp } from 'firebase/app';
import {
  getMessaging,
  getToken,
  isSupported,
  Messaging,
  onMessage,
} from 'firebase/messaging';

const firebaseConfig = {
  apiKey: process.env.REACT_APP_FIREBASE_API_KEY,
  authDomain: process.env.REACT_APP_FIREBASE_AUTH_DOMAIN,
  projectId: process.env.REACT_APP_FIREBASE_PROJECT_ID,
  storageBucket: process.env.REACT_APP_FIREBASE_STORAGE_BUCKET,
  messagingSenderId: process.env.REACT_APP_FIREBASE_MESSAGING_SENDER_ID,
  appId: process.env.REACT_APP_FIREBASE_APP_ID,
  measurementId: process.env.REACT_APP_FIREBASE_MEASUREMENT_ID,
};

// Initialize Firebase
const app = initializeApp(firebaseConfig);

let messaging: Messaging | null = null;

/**
 * Initialize Firebase Messaging
 */
const initMessaging = async () => {
  if (
    typeof window !== 'undefined' &&
    (location.protocol === 'https:' || location.hostname === 'localhost') &&
    (await isSupported())
  ) {
    messaging = getMessaging(app);
    console.log('Firebase Messaging initialized.');
  } else {
    console.warn(
      'Firebase Messaging is not supported or not running in a secure context (HTTPS or localhost).'
    );
  }
};

initMessaging();

/**
 * Request a FCM token
 */
export const requestForToken = async (): Promise<string | null> => {
  if (!messaging) {
    console.warn('Messaging is not initialized or not supported.');
    return null;
  }

  try {
    const currentToken = await getToken(messaging, {
      vapidKey: process.env.REACT_APP_FIREBASE_KEY,
    });

    if (currentToken) {
      console.log('FCM Token:', currentToken);
      return currentToken;
    } else {
      console.warn('No registration token available. Request permission.');
      return null;
    }
  } catch (err) {
    console.error('An error occurred while retrieving token:', err);
    return null;
  }
};

/**
 * Set up a listener for foreground messages
 */
export const onMessageListener = (): Promise<any> => {
  return new Promise(resolve => {
    if (!messaging) {
      console.warn('Messaging is not initialized or not supported.');
      resolve(null);
      return;
    }

    onMessage(messaging, payload => {
      console.log('Message received. ', payload);
      resolve(payload);
    });
  });
};
