// React imports
import { useEffect, useState } from 'react';

// React icon imports

// Mantine core theme hook
import { Grid, GridCol, Loader, useMantineTheme } from '@mantine/core';
import { Pagination, PaginationItem, Stack } from '@mui/material';

// Table-related imports
import { TableProps } from '../../types/table.type';

// Component imports
import {
  flexRender,
  getCoreRowModel,
  useReactTable,
} from '@tanstack/react-table';
import { useSelector } from 'react-redux';
import { getTranslation } from '../../layout/languages';

const Table = ({
  isPaginated = true,
  headerColor = '#0F783B',
  headerTextColor = '#ffffff',
  data,
  columns,
  isLoading,
  paginationInfo,
  handlePagination,
  isWithActionButton = true,
}: TableProps) => {
  const {
    rowPerPage = '0',
    totalRecords = 0,
    totalPages = 0,
    currentPage = 0,
  } = paginationInfo || {};
  const [goToPage, setGoToPage] = useState('');
  const theme = useMantineTheme();
  const table = useReactTable({
    defaultColumn: {
      size: 200, //starting column size
      minSize: 50, //enforced during column resizing
      maxSize: 500, //enforced during column resizing
    },
    data: data ?? [],
    columns,
    enableColumnResizing: true,
    columnResizeMode: 'onChange',
    getCoreRowModel: getCoreRowModel(),
    debugTable: true,
    debugHeaders: true,
    debugColumns: true,
  });

  //const { isSmallScreen, isMediumScreen, isLargeScreen } = useScreenSize(); // Using the hook to get screen size
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const [isSmallScreen, setIsSmallScreen] = useState(window.innerWidth <= 600);

  useEffect(() => {
    setIsSmallScreen(window.innerWidth <= 600);
    // eslint-disable-next-line
  }, [window.innerWidth, window.innerHeight]);
  const [selectedPage, setSelectedPage] = useState<number | null>(null);

  return (
    <>
      <section
        className={`${isPaginated ? `min-h-[${isSmallScreen ? '340px' : '370px'}]` : ''} flex justify-between flex-col mt-4 relative max-w-full overflow-x-auto md:overflow-x-visible sm:overflow-x-visible`}
      >
        {!isSmallScreen && (
          <table
            className="max-w-screen "
            style={{
              fontFamily: 'Product-Sans',
              fontSize: '12px',
            }}
          >
            <thead
              className=" h-8 lg:h-12 rounded-full"
              style={{
                color: theme.colors.darkColors[2],
                //  position: 'sticky',
                top: 0,

                zIndex: 1,
              }}
            >
              {table?.getHeaderGroups()?.map(headerGroup => (
                <tr
                  key={headerGroup.id}
                  style={{
                    borderColor: theme.colors.darkColors[3],
                  }}
                  className="tracking-widest text-skin-caption rounded-lg"
                >
                  {headerGroup?.headers?.map(header => {
                    return (
                      <th
                        key={header.id}
                        colSpan={header.colSpan}
                        style={{
                          position: 'relative',
                          width: header.getSize(),
                        }}
                        className={`text-[#ffffff] font-sans border-b border-b-[#824DDD] rounded-l-r-lg font-semibold text-[12px] capitalize ${
                          header.index === 0
                            ? 'rounded-l-xl py-4 pl-4'
                            : header.headerGroup.headers?.length - 1 ===
                                header.index
                              ? 'rounded-r-xl pl-4'
                              : 'py-3'
                        }`}
                      >
                        {header.isPlaceholder
                          ? null
                          : flexRender(
                              header.column.columnDef.header,
                              header.getContext()
                            )}
                        {header?.column?.getCanResize() && (
                          <div
                            onMouseDown={header.getResizeHandler()}
                            onTouchStart={header.getResizeHandler()}
                            className={`resizer ${
                              header.column.getIsResizing() ? 'isResizing' : ''
                            }`}
                          ></div>
                        )}
                      </th>
                    );
                  })}
                </tr>
              ))}
            </thead>
            <tbody
              style={{
                color: theme.colors.darkColors[0],
                fontSize: theme.fontSizes.md,
              }}
              className="rounded-l-lg"
            >
              {isLoading ? (
                <tr>
                  <td colSpan={columns.length} className="text-center py-8">
                    <div className="flex justify-center">
                      <Loader size="lg" color="#C28CFF" />
                    </div>
                  </td>
                </tr>
              ) : table?.getRowModel()?.rows?.length === 0 ? (
                <div className="flex min-h-[80%] font-normal mt-2 ml-2 text-[#ffffff] font-ptsans">
                  {getTranslation(language, 'dashboard', 'No data to display')}
                  ...
                </div>
              ) : (
                table?.getRowModel()?.rows?.map(row => {
                  return (
                    <tr
                      key={row.id}
                      className={`py-2 h-8 text-[#ffffff] px-8 cursor-default lg:h-10 hover:cursor-pointer ${
                        row.index % 2 === 0 ? '' : 'bg-[#4a2884]'
                      }`}
                    >
                      {row.getVisibleCells()?.map((cell, cellIndex) => {
                        return (
                          <td
                            key={cell.id}
                            style={{ width: cell.column.getSize() }}
                            className={`p-3 text-[12px] ${
                              cellIndex === 0
                                ? 'rounded-l-md'
                                : cellIndex === row.getVisibleCells().length - 1
                                  ? 'rounded-r-md'
                                  : ''
                            }`}
                          >
                            {flexRender(
                              cell.column.columnDef.cell,
                              cell.getContext()
                            )}
                          </td>
                        );
                      })}
                    </tr>
                  );
                })
              )}
            </tbody>
          </table>
        )}
        {isSmallScreen && (
          <>
            {isLoading ? (
              <tr>
                <td colSpan={columns.length} className="text-center py-8">
                  <div className="flex items-center justify-center px-32">
                    <Loader size="lg" color="#C28CFF" />
                  </div>
                </td>
              </tr>
            ) : table?.getRowModel()?.rows?.length > 0 ? (
              table?.getRowModel()?.rows?.map((row, rowIndex) => (
                <div
                  key={row.id}
                  className="card truncate rounded-lg overflow-hidden font-ptsans shadow-lg px-4 bg-[#614396] bg-opacity-5 text-[#ffffff] mt-6 relative"
                >
                  {/* ✅ Action Button at Top Right */}
                  {row.getVisibleCells()?.map((cell, index) => {
                    if (
                      index === row.getVisibleCells().length - 1 &&
                      cell.column.columnDef.header !== 'Action' &&
                      isWithActionButton
                    ) {
                      return (
                        <div
                          key={cell.id}
                          className="absolute right-4 flex justify-end text-white break-words whitespace-normal"
                        >
                          {flexRender(
                            cell.column.columnDef.cell,
                            cell.getContext()
                          )}
                        </div>
                      );
                    }
                    return null;
                  })}

                  <div className="card-info flex flex-col space-y-2">
                    {/* ✅ First Column Centered */}
                    {row.getVisibleCells()?.map((cell, index) => {
                      if (index === 0) {
                        return (
                          <div
                            key={cell.id}
                            className="flex items-center justify-center font-bold mt-8  mb-4 text-white break-words whitespace-normal"
                          >
                            {flexRender(
                              cell.column.columnDef.cell,
                              cell.getContext()
                            )}
                          </div>
                        );
                      }
                      return null;
                    })}

                    {/* ✅ Label-Value Layout excluding Action */}
                    {row.getVisibleCells()?.map((cell, index) => {
                      const header =
                        table?.getHeaderGroups()?.[0]?.headers?.[index];
                      if (!header || header.isPlaceholder) return null;

                      // Skip first column
                      if (index === 0) return null;

                      // ✅ Skip Action column in main section
                      if (
                        index === row.getVisibleCells().length - 1 &&
                        cell.column.columnDef.header !== 'Action' &&
                        isWithActionButton
                      )
                        return null;

                      if (cell.column.columnDef.header === 'Action')
                        return null;

                      return (
                        <div
                          key={cell.id}
                          className="flex justify-between items-start mb-8 text-sm text-white break-words whitespace-normal"
                        >
                          <span className="w-2/5 font-bold break-words whitespace-normal">
                            {flexRender(
                              header.column.columnDef.header,
                              header.getContext()
                            )}
                          </span>
                          <span
                            className="w-3/5 text-right break-words whitespace-normal"
                            style={{ color: '#ffffff' }}
                          >
                            {flexRender(
                              cell.column.columnDef.cell,
                              cell.getContext()
                            )}
                          </span>
                        </div>
                      );
                    })}
                  </div>
                </div>
              ))
            ) : (
              <div className="card max-w-sm min-h-auto rounded-lg overflow-hidden shadow-lg px-4 bg-color-white text-color-white mt-6">
                <div className="card-info ">
                  <div className="flex font-productsans min-h-[80%] items-center justify-center">
                    {getTranslation(
                      language,
                      'dashboard',
                      'No data to display'
                    )}
                    ...
                  </div>
                </div>
              </div>
            )}
          </>
        )}
        {isPaginated && !isSmallScreen && data?.length !== 0 && !isLoading && (
          <>
            <div className="w-full flex flex-col mb-6">
              <div className="w-full h-14 justify-center rounded-3xl  ">
                <div className="flex flex-row justify-between">
                  <div className="flex flex-row m-1.5">
                    <div className="ml-1 flex flex-row font-sans text-[14px] mt-4 font-normal text-color-white space-x-1">
                      <p>{getTranslation(language, 'dashboard', 'Showing')}</p>
                      <p>
                        {Math.min(
                          Number(rowPerPage),
                          Number(totalRecords) -
                            (Number(currentPage) - 1) * Number(rowPerPage)
                        )}
                      </p>

                      <p> {getTranslation(language, 'dashboard', 'of')}</p>
                      <p> {totalRecords}</p>
                      <p> {getTranslation(language, 'dashboard', 'Entries')}</p>
                    </div>
                  </div>
                  <div className="flex flex-row ">
                    {isSmallScreen && !isLoading ? (
                      <Grid>
                        <GridCol span={12} className="flex justify-center">
                          <Stack spacing={2}>
                            <Pagination
                              count={totalPages}
                              page={currentPage}
                              shape="rounded"
                              siblingCount={1}
                              boundaryCount={1}
                              onChange={(_, page) =>
                                handlePagination('goto', page)
                              }
                              renderItem={item => {
                                const hideSeparator =
                                  selectedPage !== null &&
                                  (item.page === selectedPage - 1 ||
                                    item.page === selectedPage);

                                // Disable previous on first page
                                if (item.type === 'previous') {
                                  return (
                                    <span
                                      {...item}
                                      style={{
                                        padding: '4px 8px',
                                        color:
                                          Number(currentPage) === 1
                                            ? '#888'
                                            : '#fff',
                                        background: 'transparent',
                                        borderRadius: '8px',
                                        cursor:
                                          Number(currentPage) === 1
                                            ? 'not-allowed'
                                            : 'pointer',
                                        fontFamily: 'Product-Sans',
                                        fontSize: '13px',
                                        fontWeight: 'normal',
                                        pointerEvents:
                                          currentPage === 1 ? 'none' : 'auto',
                                      }}
                                    >
                                      {getTranslation(
                                        language,
                                        'dashboard',
                                        'Previous'
                                      )}
                                    </span>
                                  );
                                }

                                // Disable next on last page
                                if (item.type === 'next') {
                                  return (
                                    <span
                                      {...item}
                                      style={{
                                        padding: '4px 8px',
                                        color:
                                          Number(currentPage) ===
                                          Number(totalPages)
                                            ? '#888'
                                            : '#fff',
                                        background: 'transparent',
                                        borderRadius: '8px',
                                        cursor:
                                          Number(currentPage) ===
                                          Number(totalPages)
                                            ? 'not-allowed'
                                            : 'pointer',
                                        fontFamily: 'Product-Sans',
                                        fontSize: '13px',
                                        fontWeight: 'normal',
                                        pointerEvents:
                                          Number(currentPage) ===
                                          Number(totalPages)
                                            ? 'none'
                                            : 'auto',
                                      }}
                                    >
                                      {getTranslation(
                                        language,
                                        'dashboard',
                                        'Next'
                                      )}
                                    </span>
                                  );
                                }

                                return (
                                  <div
                                    style={{
                                      display: 'flex',
                                      alignItems: 'center',
                                      position: 'relative',
                                    }}
                                  >
                                    <PaginationItem
                                      {...item}
                                      sx={{
                                        color: '#fff',
                                        background: 'transparent',
                                        fontSize: '14px',
                                        width: '35px',
                                        height: '35px',
                                        display: 'flex',
                                        alignItems: 'center',
                                        justifyContent: 'center',
                                        fontFamily: 'Product-Sans',
                                        borderRadius: '0px',
                                        fontWeight: 'normal',
                                        border: '1px solid transparent',
                                        '&.Mui-selected': {
                                          background: '#d63384 !important',
                                          color: '#fff',
                                          border: 'none',
                                          width: '36px',
                                          height: '38px',
                                        },
                                      }}
                                    >
                                      {String(item.page).padStart(2, '0')}
                                    </PaginationItem>

                                    {!hideSeparator && item.page !== 6 && (
                                      <div
                                        style={{
                                          position: 'absolute',
                                          right: '-2px',
                                          top: '50%',
                                          transform: 'translateY(-50%)',
                                          height: '25%',
                                          width: '1px',
                                          backgroundColor: '#fff',
                                          zIndex: 2,
                                        }}
                                      />
                                    )}
                                  </div>
                                );
                              }}
                              sx={{
                                display: 'flex',
                                justifyContent: 'center',
                                alignItems: 'center',
                                background: 'transparent',
                                fontFamily: 'Product-Sans',
                                height: '40px',
                                borderRadius: '8px',
                                paddingLeft: '6px',
                                border: '1px solid #fff',
                              }}
                            />
                          </Stack>
                        </GridCol>
                      </Grid>
                    ) : (
                      !isLoading && (
                        <div className="flex flex-row items-center space-x-4 mt-4 w-auto h-[40px] px-4">
                          <Stack spacing={2}>
                            <Pagination
                              count={totalPages}
                              page={currentPage}
                              shape="rounded"
                              siblingCount={1}
                              boundaryCount={1}
                              onChange={(_, page) =>
                                handlePagination('goto', page)
                              }
                              renderItem={item => {
                                const hideSeparator =
                                  selectedPage !== null &&
                                  (item.page === selectedPage - 1 ||
                                    item.page === selectedPage);

                                // Disable previous on first page
                                if (item.type === 'previous') {
                                  return (
                                    <span
                                      {...item}
                                      style={{
                                        padding: '4px 8px',
                                        color:
                                          Number(currentPage) === 1
                                            ? '#888'
                                            : '#fff',
                                        background: 'transparent',
                                        borderRadius: '8px',
                                        cursor:
                                          Number(currentPage) === 1
                                            ? 'not-allowed'
                                            : 'pointer',
                                        fontFamily: 'Product-Sans',
                                        fontSize: '13px',
                                        fontWeight: 'normal',
                                        pointerEvents:
                                          currentPage === 1 ? 'none' : 'auto',
                                      }}
                                    >
                                      {getTranslation(
                                        language,
                                        'dashboard',
                                        'Previous'
                                      )}
                                    </span>
                                  );
                                }

                                // Disable next on last page
                                if (item.type === 'next') {
                                  return (
                                    <span
                                      {...item}
                                      style={{
                                        padding: '4px 8px',
                                        color:
                                          Number(currentPage) ===
                                          Number(totalPages)
                                            ? '#888'
                                            : '#fff',
                                        background: 'transparent',
                                        borderRadius: '8px',
                                        cursor:
                                          Number(currentPage) ===
                                          Number(totalPages)
                                            ? 'not-allowed'
                                            : 'pointer',
                                        fontFamily: 'Product-Sans',
                                        fontSize: '13px',
                                        fontWeight: 'normal',
                                        pointerEvents:
                                          Number(currentPage) ===
                                          Number(totalPages)
                                            ? 'none'
                                            : 'auto',
                                      }}
                                    >
                                      {getTranslation(
                                        language,
                                        'dashboard',
                                        'Next'
                                      )}
                                    </span>
                                  );
                                }

                                return (
                                  <div
                                    style={{
                                      display: 'flex',
                                      alignItems: 'center',
                                      position: 'relative',
                                    }}
                                  >
                                    <PaginationItem
                                      {...item}
                                      sx={{
                                        color: '#fff',
                                        background: 'transparent',
                                        fontSize: '14px',
                                        width: '35px',
                                        height: '35px',
                                        display: 'flex',
                                        alignItems: 'center',
                                        justifyContent: 'center',
                                        fontFamily: 'Product-Sans',
                                        borderRadius: '0px',
                                        fontWeight: 'normal',
                                        border: '1px solid transparent',
                                        '&.Mui-selected': {
                                          background: '#d63384 !important',
                                          color: '#fff',
                                          border: 'none',
                                          width: '36px',
                                          height: '38px',
                                        },
                                      }}
                                    >
                                      {String(item.page).padStart(2, '0')}
                                    </PaginationItem>

                                    {!hideSeparator && item.page !== 6 && (
                                      <div
                                        style={{
                                          position: 'absolute',
                                          right: '-2px',
                                          top: '50%',
                                          transform: 'translateY(-50%)',
                                          height: '25%',
                                          width: '1px',
                                          backgroundColor: '#fff',
                                          zIndex: 2,
                                        }}
                                      />
                                    )}
                                  </div>
                                );
                              }}
                              sx={{
                                display: 'flex',
                                justifyContent: 'center',
                                alignItems: 'center',
                                background: 'transparent',
                                fontFamily: 'Product-Sans',
                                height: '40px',
                                borderRadius: '8px',
                                paddingLeft: '6px',
                                border: '1px solid #fff',
                              }}
                            />
                          </Stack>
                        </div>
                      )
                    )}
                  </div>
                </div>
              </div>
            </div>
          </>
        )}
      </section>
    </>
  );
};

export default Table;
