import { TextInput, TextInputProps } from '@mantine/core';
import { useDebouncedValue } from '@mantine/hooks';
import { useEffect, useState } from 'react';

interface DebouncedTextInputProps extends TextInputProps {
  debounceDelay?: number;
  onDebouncedChange?: (value: string) => void;
}

export function DebouncedTextInput({
  value: externalValue,
  onChange,
  onDebouncedChange,
  debounceDelay = 500,
  ...props
}: DebouncedTextInputProps) {
  const [internalValue, setInternalValue] = useState<string>(
    (externalValue as string) || ''
  );

  const [debounced] = useDebouncedValue(internalValue, debounceDelay);

  // Update local value if external value changes
  useEffect(() => {
    if (typeof externalValue === 'string') {
      setInternalValue(externalValue);
    }
  }, [externalValue]);

  // Trigger debounced callback
  useEffect(() => {
    onDebouncedChange?.(debounced);
  }, [debounced]);

  return (
    <TextInput
      {...props}
      value={internalValue}
      onChange={event => {
        const val = event.currentTarget.value;
        setInternalValue(val);
        onChange?.(event); // if caller wants immediate value too
      }}
    />
  );
}
