import axios from 'axios';
import store from '../redux';
import { cleanObject } from './../utils/common/function';

// API base URL
const API_BASE_URL = `${process.env.REACT_APP_BASE_URL}`;

// Common function to handle HTTP requests
const requestData = async <T>(
  method: string,
  url: string,
  data?: any,
  isFile = false,
  customHeaders: Record<string, string> = {}
) => {
  const token =
    sessionStorage.getItem('token') ?? store.getState()?.userInfo?.token;

  const currencyId =
    store.getState()?.userInfo?.preferences?.currency?.preferredCurrencyId;

  const axiosInstance = axios.create({
    baseURL: API_BASE_URL,
    headers: {
      ...(isFile ? {} : { 'Content-Type': 'application/json' }),
      Authorization: `Bearer ${token}`,
      ...(currencyId ? { 'X-Currency-Id': currencyId } : {}), // 👈 auto-attach if exists
      ...customHeaders, // allow manual overrides if needed
    },
  });

  const response = data
    ? await axiosInstance.request<T>({ method, url, data })
    : await axiosInstance.request<T>({
        method,
        url,
        data: cleanObject(data),
      });

  return response.data;
};

// File upload request
export const uploadFile = async <T>(url: string, file: File) => {
  const formData = new FormData();
  formData.append('file', file);

  return await requestData<T>('post', url, formData, true);
};

// GET request
export const fetchData = async <T>(url: string) => {
  return await requestData<T>('get', url);
};

// POST request
export const postData = async <T>(url: string, postData: any) => {
  return await requestData<T>('post', url, postData);
};

// PUT request
export const putData = async <T>(url: string, putData: any) => {
  const userDetails = store.getState()?.userInfo?.userInfo;

  if (putData instanceof FormData) {
    putData.append('updatedBy', userDetails?.userId?.toString() || '');
  } else {
    putData = { ...putData, updatedBy: userDetails?.userId?.toString() };
  }

  return await requestData<T>('put', url, putData);
};

// DELETE request
export const deleteData = async <T>(url: string) => {
  return await requestData<T>('delete', url);
};
